package mods.immibis.core.impl;


import java.util.ArrayList;
import java.util.List;

import mods.immibis.core.Config;
import mods.immibis.core.api.IIDAllocator;
import mods.immibis.core.api.IIDCallback;
import mods.immibis.core.api.IRecipeSaveHandler;
import mods.immibis.core.api.util.ErrorScreen;
import net.minecraft.block.Block;
import net.minecraft.item.Item;
import net.minecraftforge.common.Configuration;
import cpw.mods.fml.common.Loader;

public class IDAllocator implements IIDAllocator {
	
	private boolean canRequest = true;
	
	private static class Request {
		@SuppressWarnings("unused")
		public Object mod;
		public String name;
		public IIDCallback callback;
		@SuppressWarnings("unused")
		public int min, max, _default;
		public IDType type;
		
		public int id = -1;
	}
	
	private List<Request> requests = new ArrayList<Request>();
	
	private static class SavedRecipes<S,D> {
		private S initialState, finalState;
		private D diff;
		private final IRecipeSaveHandler<S,D> handler;
		private boolean saved = false;
		
		public SavedRecipes(IRecipeSaveHandler<S,D> handler) {
			this.handler = handler;
		}
		
		public void saveInitial() {
			initialState = handler.save();
		}
		public void saveFinal() {
			finalState = handler.save();
			diff = handler.diff(finalState, initialState);
			saved = true;
		}
		public void load() {
			handler.apply(diff);
		}
		public boolean isSaved() {
			return saved;
		}
		
		public static <S,D> SavedRecipes<S,D> create(IRecipeSaveHandler<S,D> handler) {
			return new SavedRecipes<S,D>(handler);
		}
	}
	
	private List<SavedRecipes<?,?>> savedRecipes = new ArrayList<SavedRecipes<?,?>>();
	private List<Runnable> recipeAdders = new ArrayList<Runnable>();

	public void allocate(IIDSet idSet) {
		canRequest = false;
		
		for(SavedRecipes<?,?> r : savedRecipes) {
			if(r.isSaved()) {
				r.load();
			}
		}
		for(Request r : requests) {
			if(r.id != -1) {
				//r.callback.unregister(r.id);
				//r.id = -1;
				throw new AssertionError("already registered: "+r.name);
			}
		}
		
		boolean hasIDResolver = Loader.isModLoaded("IDResolver") || Loader.isModLoaded("IdFix") || Config.getBoolean("core.assumeIDResolver", true);
		
		// first pass: register blocks with known IDs
		// second pass: allocate new IDs
		for(boolean firstPass : new boolean[] {true, false}) {
			for(Request r : requests) {
				if(r.id >= 0) continue; // already allocated this
				
				r.id = idSet.getIDFor(r.name, r.type, r._default, !firstPass);
				System.out.println(r.name+" "+firstPass+" "+r.id);
				if(r.id == -2) continue; // no ID set yet
				
				if(r.id != -1 && (r.id < 1 || r.id >= r.max)) {
					if(r.type == IDType.Block)
						ErrorScreen.displayFatalError("The block ID set in immibis.cfg for '"+r.name+"' is out of range.",
							(r.id < 1 ? "The minimum block ID is 1" : "The maximum block ID is 4095") + ", but the current setting is "+r.id+".",
							"Please correct the problem and restart Minecraft.");
					else if(r.type == IDType.TerrainBlock)
						ErrorScreen.displayFatalError("The block ID set in immibis.cfg for '"+r.name+"' is out of range.",
							(r.id < 1 ? "The minimum block ID is 1." : "The maximum block ID for this block is 255."),
							"The current setting is "+r.id+".",
							"Please correct the problem and restart Minecraft.");
					else
						ErrorScreen.displayFatalError("The item ID set in immibis.cfg for '"+r.name+"' is out of range.",
							(r.id < 1 ? "The minimum item ID is 1, and item IDs below 4096 are not recommended" : "The maximum item ID is 31999") + ", but the current setting is "+r.id+".",
							"Please correct the problem and restart Minecraft.");
					
					throw new AssertionError("Out-of-range ID for "+r.name+" from "+idSet);
				}
				
				if(!hasIDResolver)
				{
					Item item = Item.itemsList[r.id];
					Block block = r.id < Block.blocksList.length ? Block.blocksList[r.id] : null;
					if(block != null) {
						ErrorScreen.displayFatalError("The ID set in immibis.cfg for '"+r.name+"' is already used by",
							"'" + (block.getUnlocalizedName().equals("tile.null") ? block.toString() : block.getUnlocalizedName()) + "' (ID "+r.id+")",
							"Please correct the problem and restart Minecraft."
							);
					} else if(item != null) {
						ErrorScreen.displayFatalError("The ID set in immibis.cfg for '"+r.name+"' is already used by",
							"'" + (item.getUnlocalizedName().equals("item.null") ? item.toString() : item.getUnlocalizedName()) + "' (ID "+r.id+")",
							"Please correct the problem and restart Minecraft."
							);
					}
					if(item != null || block != null)
						throw new RuntimeException("Slot "+r.id+" already occupied by "+item+"/"+block+" when adding "+r.name);
				}
				
				if(r.id != -1) {
					r.callback.register(r.id);
					if(!hasIDResolver) {
						switch(r.type) {
						case TerrainBlock:
						case Block:
							if(Block.blocksList[r.id] == null)
								throw new AssertionError(r.callback+" failed to register block "+r.name);
							break;
						case Item:
							if(Item.itemsList[r.id] == null)
								throw new AssertionError(r.callback+" failed to register item "+r.name);
							break;
						}
					}
				}
			}
		}
		
		for(SavedRecipes<?,?> r : savedRecipes)
			if(!r.isSaved())
				r.saveInitial();
		
		for(Runnable r : recipeAdders)
			r.run();
		
		for(SavedRecipes<?,?> r : savedRecipes)
			if(!r.isSaved())
				r.saveFinal();
		
	}

	private Request request(Object mod, String name, IIDCallback callback, int min, int max, boolean allowDefault, IDType type) {
		if(!canRequest)
			throw new IllegalStateException("Too late to request IDs, use pre-init");
		
		Request r = new Request();
		r.id = -1;
		r.mod = mod;
		r.name = name;
		r.callback = callback;
		r.min = min;
		r.max = max;
		r.type = type;
		r._default = allowDefault ? min + (Math.abs(r.name.hashCode()) % (max - min)) : max - 1;
		requests.add(r);
		return r;
	}

	@Override
	public void addRecipes(Runnable callback) {
		if(!canRequest)
			throw new IllegalStateException("Too late to register recipe callbacks, use pre-init");
		recipeAdders.add(callback);
	}

	@Override
	public void registerRecipeSaveHandler(IRecipeSaveHandler<?,?> handler) {
		if(!canRequest)
			throw new IllegalStateException("Too late to register recipe save handlers, use pre-init");
		savedRecipes.add(SavedRecipes.create(handler));
	}

	@Override
	public void requestItem(Object mod, String name, IIDCallback callback) {
		Request r = request(mod, name, callback, 4096, 32000, true, IDType.Item);
		if(!Config.config.getCategory(Configuration.CATEGORY_ITEM).containsKey(r.name)) {
			Config.config.get(Configuration.CATEGORY_ITEM, r.name, r._default).getInt(r._default);
			Config.save();
		}
	}

	@Override
	public void requestTerrainBlock(Object mod, String name, IIDCallback callback) {
		Request r = request(mod, name, callback, 128, 256, false, IDType.TerrainBlock);
		if(!Config.config.getCategory(Configuration.CATEGORY_BLOCK).containsKey(r.name)) {
			Config.config.getTerrainBlock(Configuration.CATEGORY_BLOCK, r.name, r._default, null);
			Config.save();
		}
	}

	@Override
	public void requestBlock(Object mod, String name, IIDCallback callback) {
		Request r = request(mod, name, callback, 256, 4096, true, IDType.Block);
		/*ConfigCategory cat = Config.config.getCategory(Configuration.CATEGORY_BLOCK);
		
		boolean save = false;
		
		if(cat.containsKey(r.name)) {
			cat.put(r.name+".id", new Property(r.name+".id", cat.remove(r.name).getString(), Type.INTEGER));
			save = true;
		}

		if(!cat.containsKey(r.name+".id")) {
			Config.config.getBlock(r.name+".id", r._default);
			save = true;
		}
		
		if(save)
			Config.save();*/
	}

}
