package mods.immibis.redlogic.lamps;

import java.util.List;
import java.util.Random;

import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

import mods.immibis.core.api.porting.SidedProxy;
import mods.immibis.redlogic.RedLogicMod;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.client.renderer.texture.IconRegister;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.network.packet.Packet53BlockChange;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.Icon;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

public class BlockLampCube extends Block {
	
	static enum Type {Normal, Decorative, Indicator}
	
	static final int COLOURS[] = {
		0xFFFFFF,
		0xFFA100,
		0xFF00FF,
		0xAEAEFF,
		0xFFFF00,
		0xA1FF63,
		0xFFB9B9,
		0x9D9D9D,
		0xCBCBCB,
		0x00FFFF,
		0xAE00FF,
		0x0000FF,
		0xA55A00,
		0x00A600,
		0xFF0000,
		0x3B3B3B
	};
	
	boolean powered;
	Type type;
	
	static Icon iUncoloured, iColoured;
	static boolean renderingColouredPart;
	
	static int renderType = SidedProxy.instance.getUniqueBlockModelID("mods.immibis.redlogic.lamps.LampRenderStatic", true);
	
	public BlockLampCube(int id, boolean powered, Type type) {
		super(id, Material.redstoneLight);
		this.powered = powered;
		this.type = type;
		
		lightValue[blockID] = (powered && type != Type.Indicator) ? 15 : 0;
		
		if(type == Type.Decorative)
			setCreativeTab(CreativeTabs.tabDecorations);
		else if(!powered)
			setCreativeTab(CreativeTabs.tabRedstone);
		
		switch(type) {
		case Normal: setUnlocalizedName("redlogic.lamp.cube.n"); break;
		case Decorative: setUnlocalizedName("redlogic.lamp.cube.d"); break;
		case Indicator: setUnlocalizedName("redlogic.lamp.cube.i"); break;
		}
		
		setStepSound(soundGlassFootstep);
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public void registerIcons(IconRegister par1IconRegister) {
		// load the icons only once
		if(type != Type.Normal || powered)
			return;
		
		iUncoloured = par1IconRegister.registerIcon("redlogic:lamp/cube-uncoloured");
		iColoured = par1IconRegister.registerIcon("redlogic:lamp/cube-coloured");
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public Icon getIcon(int par1, int par2) {
		return renderingColouredPart ? iColoured : iUncoloured;
	}
	
	@Override
	public int getRenderType() {
		return renderType;
	}
	
	@Override
	public int damageDropped(int par1) {
		return par1;
	}
	
	@Override
	public void onBlockAdded(World par1World, int par2, int par3, int par4) {
		super.onBlockAdded(par1World, par2, par3, par4);
		onNeighborBlockChange(par1World, par2, par3, par4, 0);
	}
	
	@Override
	public void onNeighborBlockChange(World par1World, int par2, int par3, int par4, int par5) {
		if(par1World.isRemote || type == Type.Decorative)
			return;
		
		boolean nowPowered = isPowered(par1World, par2, par3, par4);
		if(nowPowered != powered)
			par1World.scheduleBlockUpdate(par2, par3, par4, blockID, 1);
	}
	
	@Override
	public void updateTick(World par1World, int par2, int par3, int par4, Random par5Random) {
		if(par1World.isRemote || type == Type.Decorative)
			return;
		
		boolean nowPowered = isPowered(par1World, par2, par3, par4);
		if(nowPowered != powered)
			par1World.setBlock(par2, par3, par4, getOtherBlockID(), par1World.getBlockMetadata(par2, par3, par4), 3);
	}
	
	private boolean isPowered(World w, int x, int y, int z) {
		return w.isBlockIndirectlyGettingPowered(x, y, z);
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public int getMixedBrightnessForBlock(IBlockAccess par1iBlockAccess, int par2, int par3, int par4) {
		if(renderingColouredPart && powered)
			return 0x00F000F0;
		return super.getMixedBrightnessForBlock(par1iBlockAccess, par2, par3, par4);
	}
	
	private int getOtherBlockID() {
		if(type == Type.Normal) {
			if(powered)
				return RedLogicMod.lampCubeOff.blockID;
			else
				return RedLogicMod.lampCubeOn.blockID;
		} else if(type == Type.Indicator) {
			if(powered)
				return RedLogicMod.lampCubeIndicatorOff.blockID;
			else
				return RedLogicMod.lampCubeIndicatorOn.blockID;
		} else {
			return blockID;
		}
	}
	
	@Override
	public int idPicked(World par1World, int par2, int par3, int par4) {
        if(type != Type.Decorative && powered)
        	return getOtherBlockID();
        else
        	return super.idPicked(par1World, par2, par3, par4);
    }
	
	@Override
	@SideOnly(Side.CLIENT)
	public int colorMultiplier(IBlockAccess par1iBlockAccess, int par2, int par3, int par4) {
		return getRenderColor(par1iBlockAccess.getBlockMetadata(par2, par3, par4));
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public int getRenderColor(int par1) {
		if(!renderingColouredPart)
			return 0xFFFFFF;
		
		try {
			int c = COLOURS[par1];
			if(!powered)
				c = (c >> 2) & 0x3F3F3F;
			return c;
		} catch(ArrayIndexOutOfBoundsException e) {
			return 0;
		}
	}
	
	public static class On extends BlockLampCube {public On(int id) {super(id, true, Type.Normal);}}
	public static class Off extends BlockLampCube {public Off(int id) {super(id, false, Type.Normal);}}
	public static class Decorative extends BlockLampCube {public Decorative(int id) {super(id, true, Type.Decorative);}}
	public static class IndicatorOn extends BlockLampCube {public IndicatorOn(int id) {super(id, true, Type.Indicator);}}
	public static class IndicatorOff extends BlockLampCube {public IndicatorOff(int id) {super(id, false, Type.Indicator);}}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	@SideOnly(Side.CLIENT)
	public void getSubBlocks(int par1, CreativeTabs par2CreativeTabs, List par3List) {
		if(powered && type != Type.Decorative)
			return;
		
		for(int k = 0; k < 16; k++)
			par3List.add(new ItemStack(this, 1, k));
	}
	
	@Override
	public boolean onBlockActivated(World w, int x, int y, int z, EntityPlayer ply, int par6, float par7, float par8, float par9) {
		ItemStack h = ply.getCurrentEquippedItem();
		if(h != null && h.itemID == Item.dyePowder.itemID) {
			if(!w.isRemote) {
				if(MinecraftServer.getServer().func_96290_a(w, x, y, z, ply)) { // isBlockProtected
					// No permission.
					if(ply instanceof EntityPlayerMP)
						((EntityPlayerMP)ply).playerNetServerHandler.netManager.addToSendQueue(new Packet53BlockChange(x, y, z, w));
					return true;
				}
			}
			
			w.setBlockMetadataWithNotify(x, y, z, 15 - h.getItemDamage(), 3);
			
			return true;
		}
		return false;
	}
}
